/*
    Copyright 2017 Hardy Lau

    Solver for Ground Conductivity and Ground Permittivity
    v3.0 27.01.2017

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.

    Dieses Programm ist Freie Software: Sie knnen es unter den Bedingungen
    der GNU General Public License, wie von der Free Software Foundation,
    Version 3 der Lizenz oder (nach Ihrer Wahl) jeder neueren
    verffentlichten Version, weiterverbreiten und/oder modifizieren.

    Dieses Programm wird in der Hoffnung, dass es ntzlich sein wird, aber
    OHNE JEDE GEWAEHRLEISTUNG, bereitgestellt; sogar ohne die implizite
    Gewaehrleistung der MARKTFAEHIGKEIT oder EIGNUNG FR EINEN BESTIMMTEN ZWECK.
    Siehe die GNU General Public License fr weitere Details.

    Sie sollten eine Kopie der GNU General Public License zusammen mit diesem
    Programm erhalten haben. Wenn nicht, siehe <http://www.gnu.org/licenses/>.

    --------------

    Compile with:     gcc -lm owl-calc.c -o owl-calc
*/

#include <math.h>
#include <complex.h>
#include <stdio.h>
#include <stdlib.h>

//Constants
#define speed_of_light 299792458.0
#define e0 8.85418781762e-12
#define mu0 1.2566370614359e-6

//Global variables with geometry of two wire line
double len,s,d,fringing_geo,k;

//------------------------

double gb( double sigma) {
 return( sigma * M_PI / acosh(s/d));
}

//------------------------

double lb() {
  return( mu0 / M_PI * acosh( s/d));
}

//------------------------

double cb( double er) {
  return( (M_PI * e0 * er) / acosh( s/d));
}

//------------------------

double rb( double f) {
  return( 1.66e-7 * k * sqrt(f) / d);
}

//------------------------

_Complex double eta( double f, double er, double sigma) {
  return( csqrt( ( rb(f) + 2 * M_PI * I * f * lb()) * ( gb(sigma) + 2.0 * M_PI * f * I * cb(er))));
}

//------------------------

double fringing_geo_calc() {
  double summe = 0;
  double D = s / d;
  int i;

  for( i = 1; i < 20; i++) {
    summe += sinh( log( D + sqrt( D * D - 1))) / sinh( (double)i * log( D + sqrt( D * D - 1)));
  }
  return( summe * d / 2.0 * M_PI);
}

//------------------------

_Complex double ze( double f, double er, double sigma) {
  return( 1.0 / ( sigma * fringing_geo + ( I * 2.0 * M_PI * f * e0 * er * fringing_geo)));
}

//------------------------

_Complex double z0( double f, double er, double sigma) {
  return( csqrt( (rb(f) + 2.0 * I * M_PI * f * lb() ) / ( gb(sigma) + 2.0 * I * M_PI * f * cb(er) )));
}

//------------------------

_Complex double z( double f, double er, double sigma, _Complex double zl, double l) {
 return( z0(f,er,sigma) * ( ( zl * ccosh( eta(f,er,sigma) * l) + z0(f,er,sigma) *
         csinh( eta(f,er,sigma) * l)) / ( zl * csinh( eta(f,er,sigma) * l) + z0(f,er,sigma) *
         ccosh( eta(f,er,sigma) * l))));
}

//------------------------

int main( int argc, char *argv[])
{
 double freq, r, jx, epsilon = 1.0, sigma = 0.0, diff_old, diff_epsilon, diff_sigma;
 _Complex double imp;
 long notendless = 500000L;
 int arg_error = 0;

 //Welcome
 printf( "Ground conductivity and relative permittivity calculator\nfor measurements with an open wire line.\n");
 printf( "Copyright 2017 Hardy Lau, DL1GLH, http://www.dl1glh.de/groundconductivity.html\n\n");

 //Arguments
 if( argc < 8) {
   printf( "Usage: %s <length[mm]> <diameter[mm]> <distance[mm]> <frequency[mc]>\n       <impedance[real, ohms]>", argv[0]);
   printf( " <impedance[imaginary, +-ohms]> <K>\n\n");
   printf( "       <K> relative resistance of the probes compared to copper:\n");
   printf( "       copper=1.0 aluminium alloyed=1.3-2.0 brass=1.9-2.3 zinc=1.9\n       steel=2.8-3.6 stainless steel=7.2\n");
   printf( "       Please, use a dot \".\" as a decimal separator.\n");
   return(1);
 }

 //Length[mm]
 if( sscanf( argv[1], "%lf", &len) == 1) {
   len /= 1000.0;
   if( len < 0.0) {
     len = fabs( len);
     arg_error++;
   }
 } else {
   len = 0.4; //default value
   arg_error++;
 }

 //Diameter[mm]
 if( sscanf( argv[2], "%lf", &d) == 1) {
   d /= 1000.0;
   if( d < 0.0) {
     d = fabs( d);
     arg_error++;
   }
 } else {
   d = 0.008; //default value
   arg_error++;
 }

 //Distance[mm]
 if( sscanf( argv[3], "%lf", &s) == 1) {
   s /= 1000.0;
   if( s < 0.0) {
     s = fabs( s);
     arg_error++;
   }
 } else {
   s = 0.048; //default value
   arg_error++;
 }

 //Distance plausibility
 if( s <= d) {
   s = d + 0.0001;
   arg_error++;
 }

 //Frequency[mc]
 if( sscanf( argv[4], "%lf", &freq) == 1) {
   freq *= 1.0e6;
   if( freq <= 0.0) {
     freq = fabs( freq);
     arg_error++;
   }
 } else {
   freq = 1.0e7; //default value
   arg_error++;
 }

 //Measured impedance
 if( ( sscanf( argv[5], "%lf", &r) == 1) && ( sscanf( argv[6], "%lf", &jx) == 1)) {
   if( r < 0.0) {
     r = fabs(r);
     arg_error++;
   }
   imp = r + I * jx;
 } else {
   imp = 30.0 - 20.0 * I; //default value
   arg_error;
 }

 //Material of probe
 if( sscanf( argv[7], "%lf", &k) == 1) {
   if( k < 1.0 || k > 50.0) {
     k = 1.0;
     arg_error++;
   }
 } else {
   k = 1.0; //default value
   arg_error++;
 }

 if( arg_error) {
   printf( "ERROR: There is something wrong with the parameters your provided!\n       Partial using corrected or default values.\n\n");
 }

 //Geometrie of fringing effect. Compute only once
 fringing_geo=fringing_geo_calc();

 //Solver
 diff_old = cabs( z( freq, epsilon, sigma, ze( freq, epsilon, sigma), len) - imp);

 while(notendless--){
   diff_epsilon = cabs( z( freq, epsilon + 0.01, sigma, ze( freq, epsilon+0.01, sigma), len) - imp);
   diff_sigma = cabs( z( freq, epsilon, sigma+0.00001, ze( freq, epsilon, sigma+0.00001), len) - imp);

   //Minimum erreicht
   if( (diff_epsilon > diff_old) && (diff_sigma > diff_old)) break;

   if( diff_epsilon <= diff_sigma) {
     epsilon += 0.01;
     diff_old = diff_epsilon;
   } else {
     sigma += 0.00001;
     diff_old = diff_sigma;
   }
 }

 //Output
 printf( "Recognized Input:\n");
 printf( "Lenght: %.1lf mm, diameter: %.1lf mm, distance: %.1lf mm, frequency: %.1lf Mc\n",
    len * 1000.0, d * 1000.0, s * 1000.0, freq / 1e6);
 printf( "Measured Impedance Zin: %.1lf%+.1lfj Ohms\n", creal( imp), cimag( imp));
 printf( "Relative resistance of the probes compared to copper: %.1lf\n\n", k); 

 printf( "Results:\n");
 if( !notendless) {
   printf( "No solution found for unknown reason.\n");
 } else {
   //Check if Epsilon is to low - Something wrong with solution
   if( diff_old > cabs( z( freq, 0.99, 0.0, ze( freq, 0.99, 0.0), len) - imp)) {
     printf( "There is something wrong with your input values or\nthe two wire line is lossless and in harmonic mode.\n\n");    
   } else {
     printf( "Zl caused by fringing effect: %.1lf%+.1lfj Ohms\n", creal( ze( freq, epsilon, sigma)), cimag( ze( freq, epsilon, sigma)));
     printf( "Impedance of two wire line in ground Z0: %.1lf%+.1lfj Ohms\n", creal( z0( freq, epsilon ,sigma)), cimag( z0( freq, epsilon ,sigma)));
     printf( "Conductivity [mS/m]: %.1lf  Relative permittivity: %.1lf\n",
        sigma * 1000.0, epsilon);
   }
 }
 return(0);
}

